//
// fgraph.hh
// 
// Made by vicen
// Login   <vicen@localhost.localdomain>
// 
// Started on  Wed Apr 19 20:34:02 2006 vicen
// Last update Thu Aug 31 17:56:02 2006 Vicen
//

/*! 
  \file 
  \brief
 */

#ifndef   	FGRAPH_HH_
# define   	FGRAPH_HH_

#include <iostream>                      // for std::cout
#include <fstream>                       // for std::cout
#include <utility>                       // for std::pair
#include <map>

// for random number generation
// #include <gsl/gsl_rng.h>
// #include <gsl/gsl_errno.h>
// #include <gsl/gsl_randist.h>

#include <boost/graph/graph_traits.hpp>
#include <boost/graph/adjacency_list.hpp>

#include "vprob.hh"
#include "pindex.hh"
#include "global.hh"

using namespace boost;

/// The factor graph class

/**
   This class uses the boost graph library and implements the BP,LR
   and COND algorithms
   
 */

class FGraph
{
protected:

  //gsl_rng *m_r;
  ///< random seed for gsl pseudorandom-number generation
  ///< to change: $ export GSL_RNG_SEED=1234

  Real m_max_factor_value;

  int m_bp_variant;  //TBPVariant
  ///< the variant used for the bp algorithm

  MethodVariant m_method;
  ///< the variant used for computing the covariances

  uint m_nfactors;
  ///< the number of factors

  uint m_nvars;
  ///< the number of variables

  uint m_nedges;
  ///< the number of edges

  MapRes m_map_res;
  ///< the multimap for the residuals will store the residuals of each edge
  ///< sorted and is indexed from the Class PEdge using an iterator
  ///< Requirements:
  ///< - Sorted Associative Container: to get the maximum in O(1)
  ///< - Pair Associative Container: edge descriptor given the max res
  ///< - Multiple Associative Container: two edges can have same res
  ///< <b>ONLY USED IN THE MAX_RES bp_variant</b>

  std::map<int, Vertex> m_map_LR_vars;
  ///< a map for obtaining the R, covs, etc...
  ///< given the Vertex it returns the row where R, covs, etc.. is located
  ///< this map is informed at each LR iteration (see doLR method)

  std::vector<Edge> m_rnd_edges;
  ///< a random sequence of the edges
  ///< <b>ONLY USED IN THE SEQ_RND bp_variant</b>

	std::string m_filename;
  ///< filename used for the input graph and for the prefix of other output filenames
  ///< as the user entered

	bool m_output;
  ///< true if output of the beliefs required

	std::string m_filename_cov;
  ///< filename used for the covariances

  std::map<int, int> m_label2idx;
  ///< for accessing the variable index  given the label
  ///< label is arbitrary and depends on the factor graph file
  ///< Given a label, returns the index in m_vars_labels

  std::vector<Vertex> m_vars_labels;
  ///< for accessing the variable node given an index
  ///< in this vector, variable nodes are sorted by label
	///< Note that indexing is NOT per variable label

  std::map<Vertex, int> m_vars_pos;
  ///< inverse of m_vars_labels
  ///< given a variable vertex, it returns the position of m_vars_labels where it appears
  ///< (needed in the exact() method)
 
  std::vector<Vertex> m_vars_sizes;
  ///< a vector of variables sorted by their sizes

  std::vector<Vertex> m_factors;
  ///< a vector of factors

  /// creates the vector of variables sorted by size
  /** details:
   */
  void buildVarsSizes();

  /// creates the vector of variables sorted by label and the map of <variables, pos>
  /** details:
   */
  void buildVarsPos();

  /// creates the vector of factors
  /** details:
   */
  void buildFactors();

  /// adds an edge to the graph
  /** \param s the source vertex
      \param t the target vertex
      \param size the size of the BP messages
      \param pos the pos in the corresponding factor vertex
   */
  void addEdge(const Vertex &s, const Vertex &t, BGraph &_g, const uint &size, const uint &pos);

public:

  Real m_tol;
  ///< the threshold for BP to converge

  uint m_max_iter;
  ///< the maximum number of iterations until BP convergence

  Real m_maxdif;
  ///< the maximum difference between two iterations
  BGraph *g;
  ///< the boost graph

  /// default constructor
  /**
   */
  FGraph() { std::cout << "no filename, exiting ..." << std::endl; exit(-1); }

  /// constructor
  /** \param filename the filename from which the factor graph is read
      \param bp_variant the variant of the bp algorithm:
   */
  FGraph(std::string filename, TBPVariant bp_variant=BP_SEQFIX);

  void exact();

  /// Destructor
  /** destroys the boost graph object
   */
  virtual ~FGraph();

  /// returns the number of factors
  /** details:
   */
  int getNFactors() { return m_nfactors; }

  /// returns the number of variables
  /** details:
   */
  int getNVars() { return m_nvars; }

  /// returns the vector sorted by label for accessing the nodes
  /** details:
   */
  std::vector<Vertex> getVarsLabels() { return m_vars_labels; }

  /// performs one BP n message update
  /** \param e The edge
      \return the new n message
   */
  VProb getBPn(const Edge& e, bool old);

  /// performs one BP m message update
  /** \param e The edge
      \return the new m message
   */
  VProb getBPm(const Edge& e, bool old);

  /// performs one LR N message update
  /** \param e The edge
      \param e The variable k
      \return the new N message
   */
  VProb getLRN(const Edge& e, const Vertex& k, const bool norm, int verb=0);

  /// performs one LR M message update
  /** \param e The edge
      \param e The variable k
      \return the new M message
   */
  VProb getLRM(const Edge& e, const Vertex& k, int verb=0);

  /// performs the LR algorithm
  /** \param tol The threshold (belief change) for convergence
      \param maxiter The maximum number of iterations for convergence
      \return true if convergence
   */
  bool doLR(Real tol, uint maxiter, int verb=0);

	/// performs the LR algorithm
  /** \param tol The threshold (belief change) for convergence
      \param maxiter The maximum number of iterations for convergence
      \return true if convergence
   */
  bool doBP(Real tol, uint maxiter, int verb=0, bool output=false);

  /// Returns the single node EXACT marginal
  /** \param index of the variable
   *  neither the label nor the Vertex. index is just the index of the m_vars_labels vector
   */
  VProb getExactMarginal(int index) { return (*g)[m_vars_labels[index]].p; }

  /// Returns the single node marginal of Belief Propagation
  /** \param index of the variable
   *  neither the label nor the Vertex. index is just the index of the m_vars_labels vector
   */
  VProb getBPMarginal(int index) { return (*g)[m_vars_labels[index]].bel; }

  /// Returns the single node marginal error of Belief Propagation
  /** Precondition: BP must be run before and the exact marginals should have been set using
   *  setExactMarginals or alternatively using the exact() method
   */
  Real getBPError() {
    m_error_bp = -1;
    for (VectorVertexIt itv = m_vars_labels.begin(); itv != m_vars_labels.end(); itv++) {
      for (uint val = 0; val < (*g)[*itv].p.size(); val++) {
        Real err = fabs( (*g)[*itv].bel[val] - (*g)[*itv].p[val] );
        if (err > m_error_bp)
          m_error_bp = err;
      }    
    }
    return m_error_bp;
  }
 
  /// Returns the logZ error of Belief Propagation
  /** Precondition: BP must be run before and m_logZ_exact should have been before
   *  or alternatively the exact() method
   */
  Real getBPZError() {
    m_error_Zbp = fabsl( m_logZ_bp - m_logZ_exact );
    //m_error_Zbp = fabsl( expl(m_logZ_bp - m_logZ_exact) - 1 );
      	  return m_error_Zbp;
  }

  /// Sets the single node marginal as if it was the exact result
  /** \param index of the variable
   *  \param prob the exact marginal
   *  neither the label nor the Vertex. index is just the index of the m_vars_labels vector
   */
  void setExactMarginal(int index, const VProb &prob) { (*g)[m_vars_labels[index]].p = prob; m_exact = true; }

  /// Sets the single node marginal as if it was the exact result
  /** \param label
   *  \param prob the exact marginal
   */
  void setExactMarginalLabel(int label, const VProb &prob) { setExactMarginal(m_label2idx[label], prob); }

  void setExactLogZ(Real logZ) { m_logZ_exact = logZ; }

	/// Returns the covariances between two nodes
  /** \param label1 the variable i
      \param label2 the variable j
   */
  VProb getCov(int label_i, int label_j);

  /// Clamp / Unclamp indexing per variable index
  std::map<Vertex, VProb> clamp(int label_i, uint value);
  void unclamp(int var, std::map<Vertex, VProb> factors);

  /// Clamp / Unclamp indexing per variable label 
	std::map<Vertex, VProb> clampLabel(int label, uint val) {
		if ( (*g)[m_vars_labels[m_label2idx[label]]].label != label ) throw "Error in variable labels";
		return clamp( m_label2idx[label], val ); 
	}
  void unclampLabel(int label, std::map<Vertex, VProb> factors) {
		if ( (*g)[m_vars_labels[m_label2idx[label]]].label != label ) throw "Error in variable labels";
		unclamp( m_label2idx[label], factors );
	}
	
  /// performs the COND algorithm
  /** details:
      \return true if all BP iterations converge
   */
  bool doCOND(Real tol, uint maxiter);

  void setTBPVariant(const TBPVariant &variant) { m_bp_variant = variant; }
  TBPVariant getBPVariant() { return (TBPVariant)m_bp_variant; }

  /// sets the BP m and n messages to 1
  /** details:
   */
  void reset();

  /// outputs the covariance matrix to a file
  /** details:
   */
  void writeCov();

  /// returns the filename where the covariances have been saved
  /** details:
   */
  std::string getFilenameCov() { return m_filename_cov; }

  std::string getBPVariantString() {
    std::string ret;
    switch (m_bp_variant) {
      case BP_MAXRES: ret = "BP_MAXRES";
      case BP_SEQRND: ret = "BP_SEQRND";
      case BP_PARAL:  ret = "BP_PARAL";
      case BP_SEQFIX: ret = "BP_SEQFIX";
    }
    return ret;
  }

  /// outputs the graph to an stream
  /** details:
   */
  friend std::ostream& operator<< (std::ostream& os, FGraph& fg);

  void graph2Dot(const char *);

  uint m_BP_steps;
  uint m_LR_steps;

  clock_t m_cpu_bp;
  clock_t m_cpu_lr;

	bool m_exact;

  Real m_logZ_exact;    // log partition function for the exact solution
  Real m_logZ_bp;       // log partition function for the bp solution

  Real m_error_bp;      // bp error in marginals
  Real m_error_Zbp;     // bp error in logZ

	bool m_negpot;        // true if at least some negative potential
};

#endif	    /* !FGRAPH_HH_ */
