//
// loopgraph.cc
//  
// Made by Vicen Gomez
// Login   <vicen@localhost.localdomain>
// 
// Started on  Tue May 16 11:04:13 2006 Vicen Gomez
//

#include "loopgraph.hh"

// testline
//#define PRINT_GRAPH         1
//#define PRINT_LOOPS         1

#define MAX_STEPS           UINT_MAX
#define LC_VERBOSE          0            // verbose each iteration (loops, queue size, etc. )
//#define DOT_LOOPS           1

void LoopGraph::loop2Dot(const MapGLoopIt &it, const char *filename)
{
  VectorVertex vars, factors;
  SetEdges setedges;
 
  // check if disconnected loop
  if ((*it).second.vloop.size() == 1) {

    // Disconnected loops: multiply vertices of all the components
    std::vector<MapGLoopIt> itg = m_bloops[(*it).first];
    for (uint l = 0; l < itg.size(); l++) {
      VectorLVertex comp = (*itg[l]).second.vloop;
      for (uint i = 0; i < comp.size(); i++) {
        if (gcore[comp[i]].type == VAR) vars.push_back(comp[i]);
        else factors.push_back(comp[i]);
        SetEdges setedges1 = (*itg[l]).second.sedges, settemp = setedges; 
        set_union(setedges1.begin(), setedges1.end(), settemp.begin(), settemp.end(), inserter(setedges, setedges.begin()));
      }
    }
  }
  else  {

    // non disconnected loops
    VectorLVertex nodes = (*it).second.vloop;
    for (uint i=0; i<nodes.size(); i++) {
     if (gcore[nodes[i]].type == VAR) vars.push_back(nodes[i]);
     else factors.push_back(nodes[i]);
     setedges = (*it).second.sedges;
    }
  }

  // create dot file
  std::ofstream fout(filename);
  fout << "graph G {" << std::endl;
  fout << "graph[size=\"100,100\"];" << std::endl;

  // var nodes 
  fout << "node[shape=circle,width=0.3,height=0.3,fixedsize=true];" << std::endl;
  std::pair<VertexIt, VertexIt> it_var = vertices(gcore); 
  while (it_var.first != it_var.second) {
    if (gcore[*it_var.first].type == VAR) {

      // check if appears in the loop
      bool found = false;
      for( size_t i = 0; (i < vars.size()) && !found; i++ ) {
        if (*it_var.first == vars[i]) 
          found = true;
      }
      fout << "\tx" << gcore[*it_var.first].label;
      if (found) 
        fout << "[style=\"setlinewidth(3)\",label=" << gcore[*it_var.first].label << "];" << std::endl;
      else 
        fout << "[style=\"setlinewidth(1)\",label=" << gcore[*it_var.first].label << "];" << std::endl;
    }
    it_var.first++;
  }

  // delete
  fout << "\tx6[style=\"setlinewidth(1)\",label=6];" << std::endl;

  // factor nodes
  fout << "node[shape=box,width=0.3,height=0.3,fixedsize=true];" << std::endl;
  std::pair<VertexIt, VertexIt>  sit = vertices(gcore); 
  while (sit.first != sit.second) {
    if (gcore[*sit.first].type == FACTOR) {

      // check if appears in the loop
      bool found = false;
      for( size_t i = 0; (i < factors.size()) && !found; i++ ) {
        if (*sit.first == factors[i])
          found = true;
      }
      fout << "\tp" << gcore[*sit.first].label;
      if (found) 
        fout << "[style=\"setlinewidth(3)\",label=" << char(97+gcore[*sit.first].label) << "];" << std::endl;
      else 
        fout << "[style=\"setlinewidth(1)\",label=" << char(97+gcore[*sit.first].label) << "];" << std::endl;
    }
    sit.first++;
  }

  // delete
  fout << "\tp5[style=\"setlinewidth(1)\",label=f];" << std::endl;

  // rest
  EdgeIt ei, ei_end;
  for (tie(ei, ei_end) = edges(gcore); ei != ei_end; ei++) {
    Edge e = *ei;
    if ( setedges.find( gcore[e].label ) == setedges.end() ) {
      if ((gcore)[source(e, gcore)].type == VAR)
        fout << "\tx" << (gcore)[source(e, gcore)].label << " -- p" << (gcore)[target(e, gcore)].label << "[style=\"setlinewidth(1)\"];" << std::endl;
      else
        fout << "\tx" << (gcore)[target(e, gcore)].label << " -- p" << (gcore)[source(e, gcore)].label << "[style=\"setlinewidth(1)\"];" << std::endl;
    }
    else {
      if ((gcore)[source(e, gcore)].type == VAR)
        fout << "\tx" << (gcore)[source(e, gcore)].label << " -- p" << (gcore)[target(e, gcore)].label << "[style=\"setlinewidth(6)\"];" << std::endl;
      else
        fout << "\tx" << (gcore)[target(e, gcore)].label << " -- p" << (gcore)[source(e, gcore)].label << "[style=\"setlinewidth(6)\"];" << std::endl;
    }
  }
  fout << "\tx6 -- p4[style=\"setlinewidth(1)\"];" << std::endl;
  fout << "\tx5 -- p5[style=\"setlinewidth(1)\"];" << std::endl;

  fout << "}" << std::endl;
  fout.close();
}

void LoopGraph::buildDotFile(const VectorLVertex &vars, const VectorLVertex &factors, const SetEdges &setedges, const char *filename)
{
  // create dot file
  std::ofstream fout(filename);
  fout << "graph G {" << std::endl;
  fout << "graph[size=\"100,100\"];" << std::endl;
    
  // var nodes
  fout << "node[shape=circle,width=0.4,fixedsize=true];" << std::endl;
  for( size_t i = 0; i < vars.size(); i++ ) {
    fout << "\tx" << vars[i] << ";" << std::endl;
  }

  // factor nodes
  fout << "node[shape=box,style=filled,color=lightgrey,width=0.3,height=0.3,fixedsize=true];" << std::endl;
  for( size_t I = 0; I < factors.size(); I++ ) {
    fout << "\tp" << factors[I] << ";" << endl;
  }

  // edges
  for( SetEdgesIt it = setedges.begin(); it != setedges.end(); it++) {
    Edge e = m_edges[*it];
    if ((gcore)[source(e, gcore)].type == VAR)
      fout << "\tx" << source(e, gcore) << " -- p" << target(e, gcore) << ";" << std::endl;
    else
      fout << "\tx" << target(e, gcore) << " -- p" << source(e, gcore) << ";" << std::endl;
  }
  fout << "}" << std::endl;
  fout.close();
}

void ptabs(uint tabs)
{
  for (uint i=0; i<tabs; i++)
    std::cout << "   ";
}

void LoopGraph::printLoop(VectorInt c)
{
  for (uint i = 0; i < c.size(); i++) {
    std::cout << c[i] << m_edges[c[i]] << "  ";
  }
  std::cout << endl;
}

LoopGraph::~LoopGraph()
{
}

void LoopGraph::countSLoopsDFS(const Vertex &current,      // current vertex
                               const Vertex &parent,       // parent vertex of current
                               const Vertex &root,         // root vertex where
                               VectorInt *loop_edges,      // partial loop (edges)
                               VectorLVertex *loop_vertex, // partial loop (vertices)
                               int mark,                   // last vertex mark
                               int tabs,                   // number of tabs
                               int bound,                  // current bound
                               bool verb)                  // verbose?
{
  // prune when the maximum number of sloops (generator set) is reached
  if ( m_loops.size() >= m_S ) return;

  if ( (!bound) && (current == root) ) {

    // loop found
    VectorInt loope(*loop_edges);
    VectorLVertex loopv(*loop_vertex);

    // sort the list of vertices and the list of edges
    SetEdges sloope(SetEdges(loope.begin(), loope.end()));
    sort(loopv.begin(), loopv.end());

    // new getkey
    std::string key = createKey(sloope);

    // add the loop
    if (m_sloops.find(key) == m_sloops.end()) {
      addSLoop(key, loopv, sloope);
    }
  }
  else if (bound) {
    if (verb) { ptabs(tabs); std::cout << "node " << current << " bound = " << bound << std::endl; }

    // go recursively forall adjacent vertices
    m_marks[current] = mark++;

    OutEdgeIt ib, ie;
    for (tie(ib, ie) = out_edges(current, gcore); ib != ie; ib++) {

      Vertex next = target(*ib, gcore);
      if ( ( (m_marks[next] < 0) || ( (bound == 1) && (next == root) ) ) && (next != parent) ) {

        // do
      	(*loop_edges).push_back((gcore)[*ib].label);
      	(*loop_vertex).push_back(next);
      
      	countSLoopsDFS(next, current, root, loop_edges, loop_vertex, mark, tabs+1, bound-1, verb);
      
      	// undo
      	(*loop_vertex).pop_back();
      	(*loop_edges).pop_back();
      }
    }
    m_marks[current] = -1;
    std::cout.flush();
  }
}

void LoopGraph::preprocessGraph()
{
  // remove nodes with degree one 
  std::cout << "preprocessing graph ...";
  std::pair<VertexIt, VertexIt> it = vertices(gcore);
  uint nvertices = num_vertices(gcore);
  uint rem = 0, remp;
  do {
    remp = rem;
    std::pair<VertexIt, VertexIt> it = vertices(gcore);
    while ( (it.first != it.second) && (rem == remp) ) {
      if (degree(*it.first, gcore) == 1) {
        if (gcore[*it.first].type == FACTOR) m_nfactors--;
          clear_vertex(*it.first, gcore);

        // remove_vertex invalidates iterator
       	remove_vertex(*it.first, gcore);
       	rem++;
      }
      it.first++;
    }
  } while (rem - remp);
  
  std::cout << " " << rem << "/" << nvertices;
  std::cout << " vertices removed during preprocessing" << std::endl;

  std::ofstream fout("prep_graph.fg");
  if (fout.is_open()) {
    fout << (*this);
    fout.close();
  }
  else {
    std::cout << "error writing preprocessed graph" << std::endl;
  }
}

void LoopGraph::countSLoops()
{
  // unmark all vertices
  m_marks.clear();
  m_marks.resize(num_vertices(gcore), -1);

  // create the temporal simple-loop
  VectorInt loop_edge;
  VectorLVertex loop_vertex;

  std::cout << "counting simple loops ...";
  uint step = 0;
  for (uint bound = 3, nloops = 0; (bound <= num_vertices(gcore)) && (m_sloops.size() < m_S); bound++) {

    std::cout << bound << " "; std::cout.flush();

    uint visited = 0;
    bool verb = false;
    std::pair<VertexIt, VertexIt> vi = vertices(gcore);
    while ( visited < num_vertices(gcore) ) {

      Vertex v = *vi.first, w = *vi.second;
      countSLoopsDFS(v, w, v, &loop_edge, &loop_vertex, 0, 1, bound, verb);
      vi.first++;
      visited++;
    }

    // if loops found add them in the size vector
    if (m_sloops.size() != nloops) {

      if (bound >= m_vsizes.size())
        m_vsizes.resize(bound+1, 0);
      m_vsizes[bound] = m_sloops.size() - nloops;
      nloops = m_sloops.size();
      step++;
      m_max_length_found = bound;
    }
  }

  // set bound if the required simple-loops have been reached
  if (!m_max_length)
    m_max_length = m_max_length_found;
  std::cout << "max_length = " << m_max_length << std::endl;

  m_steps = step;
  if (!m_sloops.size()) {
    std::cout << "No simple loops found" << std::endl;
  }
  std::cout << std::endl;
  std::cout << "Max length where loops found " << m_max_length_found;
  std::cout << "/ Max step " << m_steps << std::endl;
}

uint LoopGraph::mergeLoops(const MapGLoopIt &loop1,
                           const MapGLoopIt &loop2,
                           std::string &total_key,
                           VectorLVertex &new_vloop,
                           MapGLoop &new_map,
                           const bool &verb)
{
  // fast lookup (if the total_key already exists don't merge)
  SetEdges seunion_total;
  set_union((*loop1).second.sedges.begin(), (*loop1).second.sedges.end(), (*loop2).second.sedges.begin(), (*loop2).second.sedges.end(), inserter(seunion_total, seunion_total.begin()));
  if (seunion_total.size() > m_max_length) return 0;

  uint newloops = 0;
  total_key = createKey(seunion_total);

  if (m_gloops.find(total_key) == m_gloops.end())  {

    std::string key1 = (*loop1).first, key2 = (*loop2).first;
    VectorLVertex vloop1 = (*loop1).second.vloop, vloop2 = (*loop2).second.vloop;

    // components of loop1 (vertices)
    std::vector<std::set<LVertex> > svcomp1;
    if (vloop1.size() > 1) {
      svcomp1.push_back(std::set<LVertex>(vloop1.begin(), vloop1.end()));
    }
    else {
      for (uint c = 0; c < m_bloops[key1].size(); c++) {
        svcomp1.push_back(std::set<LVertex>((*m_bloops[key1][c]).second.vloop.begin(), (*m_bloops[key1][c]).second.vloop.end()));
      }
    }
 
    // components of loop1 (edges)
    std::vector<std::set<int> > secomp1;
    if (vloop1.size() > 1) {
      secomp1.push_back((*loop1).second.sedges);
    }
    else {
      for (uint c = 0; c < m_bloops[key1].size(); c++) {
        secomp1.push_back((*m_bloops[key1][c]).second.sedges);
      }
    }
    
    // components of loop2 (vertices)
    std::vector<std::set<LVertex> > svcomp2;
    if (vloop2.size() > 1) {
      svcomp2.push_back(std::set<LVertex>(vloop2.begin(), vloop2.end()));
    }
    else {
      for (uint c = 0; c < m_bloops[key2].size(); c++) {
        svcomp2.push_back(std::set<LVertex>((*m_bloops[key2][c]).second.vloop.begin(), (*m_bloops[key2][c]).second.vloop.end()));
      }
    }
    
    // components of loop2 (edges)
    std::vector<std::set<int> > secomp2;
    if (vloop2.size() > 1) {
        secomp2.push_back((*loop2).second.sedges);
    }
    else {
      for (uint c = 0; c < m_bloops[key2].size(); c++) {
        secomp2.push_back((*m_bloops[key2][c]).second.sedges);
      }
    }
    
    // verb
    if (verb) {
      std::cout << std::endl;
      std::cout << "loop1 components:" << std::endl;
      for (uint c = 0; c < svcomp1.size(); c++) {
        std::cout << createKey(secomp1[c]) << "\t";
        for (std::set<LVertex>::iterator sit = svcomp1[c].begin(); sit != svcomp1[c].end(); sit++) std::cout << *sit << " ";
        std::cout << std::endl;
      }
      
      std::cout << "loop2 components:" << std::endl;
      for (uint c = 0; c < svcomp2.size(); c++) {
        std::cout << createKey(secomp2[c]) << "\t";
        for (std::set<LVertex>::iterator sit = svcomp2[c].begin(); sit != svcomp2[c].end(); sit++) std::cout << *sit << " ";
        std::cout << std::endl;
      }
    }
    
    // merge them
    std::vector<std::set<LVertex> > svcompf(svcomp1);
    std::vector<std::set<int> > secompf(secomp1);
    std::vector<std::set<LVertex> >::iterator itvcompf;
    std::vector<std::set<int> >::iterator itecompf;
    
    // foreach component of the second loop
    for (uint c2 = 0; c2 < svcomp2.size(); c2++) {
      std::set<LVertex> svunion(svcomp2[c2]);
      std::set<int> seunion(secomp2[c2]);
      bool deleted = true;
      
      // while no changes
      while (deleted) {
	
        deleted = false;
        itvcompf = svcompf.begin();
        itecompf = secompf.begin();
        
        // foreach component of the present merging
        while ((itvcompf != svcompf.end()) && (!deleted) ) {
          
          std::set<LVertex> sinters;
          set_intersection((*itvcompf).begin(), (*itvcompf).end(), svcomp2[c2].begin(), svcomp2[c2].end(), inserter(sinters, sinters.begin()));
          if (sinters.size()) {
            
            // if intersection is not null get the new component
            std::set<LVertex> svuniont(svunion);
            std::set<int> seuniont(seunion);
            set_union((*itvcompf).begin(), (*itvcompf).end(), svuniont.begin(), svuniont.end(), inserter(svunion, svunion.begin()));
            set_union((*itecompf).begin(), (*itecompf).end(), seuniont.begin(), seuniont.end(), inserter(seunion, seunion.begin()));
            
            // delete the element and end the loop
            svcompf.erase(itvcompf);
            secompf.erase(itecompf);
            
            deleted = true;
          }
          
          itvcompf++;
          itecompf++;
        }
      }
      if (svunion.size()) {
	
        // add the union
        svcompf.push_back(svunion);
        secompf.push_back(seunion);
      }
    }
    
    // verb
    if (verb) {
      std::cout << "final components:" << std::endl;
      for (uint c = 0; c < svcompf.size(); c++) {
        std::cout << createKey(secompf[c]) << "\t";
        for (std::set<LVertex>::iterator sit = svcompf[c].begin(); sit != svcompf[c].end(); sit++) std::cout << *sit << " ";
        std::cout << std::endl;
      }
    }

    if (svcompf.size() > 1) {
      if (verb) {
        std::cout << "disconnected loop" << std::endl;
        std::cout << "newloops " << newloops << " " << new_map.size() << " " << m_gloops.size() << std::endl;
      }

      // disconnected loop
      if (verb) {
        std::cout << "result is disconnected" << std::endl;
        for (uint c = 0; c < svcompf.size(); c++) {
          std::cout << createKey(secompf[c]) << "\t";
          for (std::set<LVertex>::iterator sit = svcompf[c].begin(); sit != svcompf[c].end(); sit++) std::cout << *sit << " ";
          std::cout << std::endl;
        }
      }

      // get the vertices of the total_loop
      std::set<LVertex> new_sloop;
      std::set<int> total_skey;
      for (uint c = 0; c < svcompf.size(); c++) {
        std::set<LVertex> svuniont(new_sloop);
        std::set<int> seuniont(total_skey);
        set_union(svuniont.begin(), svuniont.end(), svcompf[c].begin(), svcompf[c].end(), inserter(new_sloop, new_sloop.begin()));
        set_union(seuniont.begin(), seuniont.end(), secompf[c].begin(), secompf[c].end(), inserter(total_skey, total_skey.begin()));
      }
      new_vloop = std::vector<LVertex>(new_sloop.begin(), new_sloop.end());

      // check that all components are already added as gloops
      std::vector<MapGLoopIt> itcomps;
      for (uint c = 0; c < svcompf.size(); c++) {
        std::string key_comp = createKey(secompf[c]);
        MapGLoopIt loopit;
        MapGLoopIt itsloop = m_sloops.find(key_comp);
        MapGLoopIt itgloop = m_gloops.find(key_comp);
        bool add = false;
        if ( (itsloop != m_sloops.end()) && (itgloop != m_gloops.end()) ) {
          std::cout << "component found in both loop structs" << std::endl;
          exit(-1);
        }
        else if ( (itsloop != m_sloops.end()) && (itgloop == m_gloops.end()) ) loopit = itsloop;
        else if ( (itsloop == m_sloops.end()) && (itgloop != m_gloops.end()) ) loopit = itgloop;
        else if ( (itsloop == m_sloops.end()) && (itgloop == m_gloops.end()) ) add = true;
        
        std::vector<LVertex> theloop(svcompf[c].begin(), svcompf[c].end());
        
        if (add) {
          loopit = addGLoop(key_comp, theloop, secompf[c], new_map);
          m_ncnbloops++;
        
          newloops++;
          if (verb) {
            std::cout << "addGLoop" << std::endl;
            std::cout << "newloops " << newloops << " " << new_map.size() << " " << m_gloops.size() << std::endl;
          }
        }
        
        itcomps.push_back(loopit);
      }
      
      // get the cloops
      // pick one component and count cLoops recursively
      // do this for each component of the final disconnected loop
      
      for (uint c = 0; c < svcompf.size(); c++) {

        std::vector<std::set<LVertex> > svcompft(svcompf);
       	std::vector<std::set<int> > secompft(secompf);
       	std::vector<MapGLoopIt> itcompst(itcomps);
       
       	std::vector<std::set<LVertex> >::iterator itv = svcompft.begin();
       	std::vector<std::set<int> >::iterator ite = secompft.begin();
       	std::vector<MapGLoopIt>::iterator itc = itcompst.begin();
       
       	if ( (svcompft.size() != secompft.size()) || (secompft.size() != itcompst.size()) ) {
       	  std::cout << "different sizes!!" << std::endl;
       	}
       
       	// pick up the main component
       	for (uint cvt = 0; cvt < c; cvt++, itv++, ite++, itc++);
       	std::set<LVertex> svcurrent = *itv;
       	std::set<int> securrent = *ite;
       	std::vector<LVertex> vcurrent(std::vector<LVertex>((*itv).begin(), (*itv).end()));
       	std::string key_comp = createKey(std::vector<int>((*ite).begin(), (*ite).end()));
       
       	// erase the main component from the temporal vectors
       	itcompst.erase(itc);
       	svcompft.erase(itv);
       	secompft.erase(ite);
       
       	// add the disconnected loop
       	// total_key has all the edges of the union
        newloops += addDisconnectedLoop(false, vcurrent, securrent, key_comp, itcompst, total_key, seunion_total, new_map);
       	if (verb) {
       	  std::cout << "addDisconnectedLoop" << std::endl;
       	  std::cout << "newloops " << newloops << " " << new_map.size() << " " << m_gloops.size() << std::endl;
       	}

        // if the length is strictly less than the maximum length
        if ( (seunion_total.size() < m_max_length) && (m_C > 0) ) {
   
          // get the cloops recursively with the main component fixed
          for (std::set<LVertex>::iterator uit = svcurrent.begin(); uit != svcurrent.end(); uit++) {

            // unfold the first iteration for incremental calculation of the loop term
            LVertex u = *uit;
            
            OutEdgeIt u_begin, u_end;
            for (tie(u_begin, u_end) = out_edges(u, gcore); u_begin != u_end; u_begin++) {

              // if the new vertex does not belong to the main component
              LVertex v = target(*u_begin, gcore);
       	      if (svcurrent.find(v) == svcurrent.end()) {
       
       	        // recursively find cloops
       	        svcurrent.insert(v);                      // insert the vertex in the current set of vertices
       
       	        // add edges to the current loop
       	        securrent.insert(gcore[*u_begin].label);      // insert the edge in the current set of edges
       	        total_skey.insert(gcore[*u_begin].label);        // insert the edge in the total set of edges
       
                if ( LC_VERBOSE == 1) std::cout << "["; std::cout.flush();
         
                uint old_max_length = m_max_length;
                m_max_length = total_skey.size();
       
       	        newloops += countCLoopsDFS(v,
                                           u,
                                           *u_begin,
                                           svcurrent,
                                           securrent,
                                           svcompft,
                                           secompft,
                                           total_skey,
                                           itcompst,
                                           new_map,
                                           0,//new_ploopterm,
                                           1,
                                           1,
                                           1,
                                           false);
       	        if (verb) {
                  std::cout << "countCLoopsDFS" << std::endl;
                  std::cout << "newloops " << newloops << " " << new_map.size() << " " << m_gloops.size() << std::endl;
       	        }
                if ( LC_VERBOSE == 1 ) std::cout << "]"; std::cout.flush();
       
      	        m_max_length = old_max_length;
       
       	        // erase edges of the current loop
       	        total_skey.erase(gcore[*u_begin].label);        // erase the edge in the total set of edges
       	        securrent.erase(gcore[*u_begin].label);	  // erase the edge in the current set of edges
       	        svcurrent.erase(v);                          // erase the vertex in the current set of vertices
       	      }
       	    }
       	  } // end of a component
        } // end of all components
      } // end of length check
    } // end if disconnected
    else {
      
      // not a disconnected loop (everything is in svcompf[0] and secompf[0]
      total_key = createKey(secompf[0]);
      new_vloop = std::vector<LVertex>(svcompf[0].begin(), svcompf[0].end());
      if (m_gloops.find(total_key) == m_gloops.end()) {
        addGLoop(total_key, new_vloop, secompf[0], new_map);
        m_ncnbloops++;
        newloops++;
      }
    }
  }

  return newloops;
}

uint LoopGraph::countCLoopsDFS(const LVertex &u,            // current vertex
			       const LVertex &p,                              // parent of the current vertex
			       const Edge &pe,                                // parent's edge
			       std::set<LVertex> &svcurrent,                  // vertices of the current component
			       std::set<int> &securrent,                      // edges of the current component
			       std::vector<std::set<LVertex> > &svrest,       // vertices of the rest of components
			       std::vector<std::set<int> > &serest,           // edges of the rest of components
			       std::set<int> &setotal,                        // total set of edges 
			       const std::vector<MapGLoopIt> &itrest,         // iterators of the rest of the components
			       MapGLoop &new_map,                             // map to insert the new loops
			       Real ploopterm,                                // partial term of the loop
			       uint length,                                   // length of the path connecting components
			       const uint &nconnected,                        // number of current connected components
			       const uint &ntabs,                             // tabs
			       bool deb)                                      // debug?
{
  // prune loops with too large length
  if (setotal.size() > (m_max_length + m_C)) 
    return 0;
  uint ret = 0;

  // search u in the existing components
  bool found = false;
  uint c = 0;
  while ( (c < svrest.size()) && (!found) ) {
    if (svrest[c].find(u) != svrest[c].end()) found = true;
    else c++;
  }

  //  m_marks[v] = 1;
  if (found) {

    // a cloop has been found
    if (deb) { ptabs(ntabs);
      std::cout << "*******   cloop found!   ******* " << nconnected << " " << svrest.size() << std::endl;
    }

    // connect the remaining components (vertices and edges)
    std::set<LVertex> vswap(svrest[c].begin(), svrest[c].end());
    svcurrent.insert(vswap.begin(), vswap.end());
    svrest[c].clear();
    std::set<int> eswap(serest[c].begin(), serest[c].end());
    for (std::set<int>::iterator ite = serest[c].begin(); ite != serest[c].end(); ite++) {
      if (securrent.find(*ite) != securrent.end()) {
        std::cout << "strange... edge found" << std::endl;
      }
    }
    securrent.insert(eswap.begin(), eswap.end());
    serest[c].clear();

    std::string new_key_comp = createKey(securrent);
    std::string new_key_total = createKey(setotal);

    // check if the new loop is disconnected
    if (nconnected < svrest.size()) {
      if (deb) { ptabs(ntabs); std::cout << "(disconnected loop) still keep searching!" << std::endl; }

      // the cloop is disconnected

      // create vector of components
      std::vector<MapGLoopIt> new_comps;
      for (uint ci = 0; ci < svrest.size(); ci++)
        if (svrest[ci].size()) new_comps.push_back(itrest[ci]);
      std::vector<LVertex> vcurrent(svcurrent.begin(), svcurrent.end());

      if (deb) {
        std::cout << "there are " << new_comps.size() << " components to add" << std::endl;
        for (uint ci = 0; ci < svrest.size(); ci++)
          std::cout << (*new_comps[ci]).first << std::endl;
      }

      // add the main component if necessary

      // add the Disconnected loop
      for (uint cc = 0; cc < new_comps.size(); cc++) {
        if ((*new_comps[cc]).second.vloop.size() <= 1) {
          std::cout << "in countCLoopsDFS" << std::endl;	      
        }
      }

      ret += addDisconnectedLoop(true, vcurrent, securrent, new_key_comp, new_comps, new_key_total, setotal, new_map);

      // recursive Disconnected
      for (std::set<LVertex>::iterator wit = svcurrent.begin(); wit != svcurrent.end(); wit++) {

        // unfold the first iteration for incremental calculation of the loop term
        LVertex w = *wit;

        OutEdgeIt w_begin, w_end;
        for (tie(w_begin, w_end) = out_edges(w, gcore); w_begin != w_end; w_begin++) {
          LVertex x = target(*w_begin, gcore);
       
       	  // if the new vertex does not belong to the main component
       	  // and appears in the 2-core
          if ( svcurrent.find(x) == svcurrent.end() ) {
          
            // recursively find cloops
            svcurrent.insert(x);                      // insert the vertex in the current set of vertices
            
            // add edges to the current loop
            securrent.insert(gcore[*w_begin].label);      // insert the edge in the current set of edges
            setotal.insert(gcore[*w_begin].label);        // insert the edge in the total set of edges
            
            ret += countCLoopsDFS(x,
                                  w,
                                  *w_begin,
                                  svcurrent,
                                  securrent,
                                  svrest,
                                  serest,
                                  setotal,
                                  itrest,
                                  new_map,
                                  0, //new_new_ploopterm,
                                  length + 1,
                                  nconnected + 1,
                                  ntabs + 1,
                                  deb);
            
            // erase the edges and the vertex
            setotal.erase(gcore[*w_begin].label);          // erase the edge in the total set of edges
            securrent.erase(gcore[*w_begin].label);	  // erase the edge in the current set of edges
            svcurrent.erase(x);                           // erase the vertex in the current set of vertices
          }
        }
      }
    }
    else {

      // the cloop is not disconnected (build key and vloop)
      std::vector<LVertex> vcurrent(svcurrent.begin(), svcurrent.end());
      if (m_gloops.find(new_key_total) == m_gloops.end()) {
        addGLoop(new_key_total, vcurrent, setotal, new_map);
        m_cnbloops++;
        char buff[100]; sprintf(buff, "cnbloop%d.dot", m_cnbloops);
        ret++;
      }
    }

    // disconnect the remaining components
    serest[c].insert(eswap.begin(), eswap.end());
    for (std::set<int>::iterator eit = eswap.begin(); eit != eswap.end(); eit++) {
      std::set<int>::iterator ite = securrent.find(*eit);
      if (ite == securrent.end()) {
        std::cout << "edge not found!" << std::endl;
        exit(-1);
      }
      else securrent.erase(ite);
    }
    
    svrest[c].insert(vswap.begin(), vswap.end());
    for (std::set<LVertex>::iterator eit = vswap.begin(); eit != vswap.end(); eit++)
      svcurrent.erase(svcurrent.find(*eit));
  }
  else {

    // no cloop is found (keep searching recursively)
    if (deb) { ptabs(ntabs); std::cout << "keep searching" << std::endl; }

    OutEdgeIt u_begin, u_end;
    for (tie(u_begin, u_end) = out_edges(u, gcore); u_begin != u_end; u_begin++) {
      LVertex v = target(*u_begin, gcore);

      // if the new vertex does not belong to the main component
      // and belongs to the 2-core
      if ( svcurrent.find(v) == svcurrent.end() ) {
      
        // push the vertex
        svcurrent.insert(v);
        
        // add edges to the current loop
        securrent.insert(gcore[*u_begin].label);      // insert the edge in the current set of edges
        setotal.insert(gcore[*u_begin].label);        // insert the edge in the total set of edges
        
        // recursive simple (length is one more)
        ret += countCLoopsDFS(v,
                              u,
                              *u_begin,
                              svcurrent,
                              securrent,
                              svrest,
                              serest,
                              setotal,
                              itrest,
                              new_map,
                              0,//new_ploopterm,
                              length + 1,
                              nconnected,
                              ntabs + 1,
                              deb);
        
        // erase edges and vertex
        setotal.erase(gcore[*u_begin].label);        // erase the edge in the total set of edges
        securrent.erase(gcore[*u_begin].label);	  // erase the edge in the current set of edges
        svcurrent.erase(v);                          // erase the vertex in the current set of vertices
      }
    }
  }

  // return the number of new created loops
  return ret;
}

void LoopGraph::countGLoops()
{
  m_vgsizes_vertices.clear();  m_vgsizes_vertices.resize(m_vsizes.size(), 0);
  m_vgsizes_edges.clear();     m_vgsizes_edges.resize(m_vsizes.size(), 0);
  m_vbsizes_vertices.clear();  m_vbsizes_vertices.resize(m_vsizes.size(), 0);
  m_vbsizes_edges.clear();     m_vbsizes_edges.resize(m_vsizes.size(), 0);
  m_marks.clear();
  m_marks.resize(num_vertices(gcore), -1);

  std::cout << "counting non-simple generalized loops ... " << (m_sloops.size()*(m_sloops.size()-1))/2 << " combinations " << std::endl;
  std::cout << "max_length is " << m_max_length << std::endl;

  VectorLVertex loop;
  std::string key;

  // get the first generalized loops from pairs of single loops (first iteration)
  uint comp = 0, newgloops = 0, oldgloops = 0;
  MapGLoop new_loops;

  MapGLoopIt it_ = m_sloops.begin();

  // forall simple loops
  for (MapGLoopIt it1 = m_sloops.begin(); it1 != m_sloops.end(); it1++) {

    // forall simple loops
    MapGLoopIt it2 = it1; it2++;
    for (;it2 != m_sloops.end(); it2++) {

      loop.clear();
      newgloops += mergeLoops(it1, it2, key, loop, new_loops);
      if (newgloops != new_loops.size()) {
        std::cout << newgloops << "!=" << new_loops.size() << std::endl;
        exit(-1);
      }
      comp++;

      if (comp%10000000 == 0) {
	oldgloops = newgloops - oldgloops;
	std::cout << "\t" << comp/1000000 << "e6 combinations (new " << newgloops << ")(last " << oldgloops << ")" << std::endl;
	oldgloops = newgloops;
      }
    }
  }
  std::cout << "\t\tnew loops " << newgloops << " (" << m_bloops.size() << " disconnected)" << std::endl;

  // iterate merging the simple loops with the new ones at each iteration
  // until no new generalized loops are created
  new_loops = m_gloops;
  bool end = (new_loops.size() == 0);
  do {

    /*cout << "ONLY ONE ITERATION!!!!!!!!!!!!!" << endl;
    cout << "-------------------------------" << endl;
    break;*/

    MapGLoop old_loops = new_loops;
    new_loops.clear();

    unsigned long comp = 0, newgloops = 0, oldgloops = 0, oldbloops = m_bloops.size(), newbloops = 0;
    std::cout << "number of comp " << m_sloops.size() << "*" << old_loops.size();
    std::cout << " = " << m_sloops.size()*old_loops.size() << std::endl;

    VectorLVertex loop;
    std::string key;

    // prune the search if too many combinations...
    int tt = 0;
    if ( (old_loops.size() * m_sloops.size()) < 50000000 ) {
    
      // forall simple loops
      for (MapGLoopIt ms_it = m_sloops.begin(); ms_it != m_sloops.end(); ms_it++) {

        // forall generated loops in the previous iteration
        for (MapGLoopIt mg_it = old_loops.begin(); mg_it != old_loops.end(); mg_it++) {
 
          // get the generated loop
          loop.clear();
          MapGLoopIt safe_loop = m_gloops.find((*mg_it).first);
          if (safe_loop != m_gloops.end()) {

            // merge the generated loop with the simple loop
            if ( ((*ms_it).second.sedges.size() < m_max_length) && ((*safe_loop).second.sedges.size() < m_max_length) ) {
              newgloops += mergeLoops(ms_it, safe_loop, key, loop, new_loops);
              if (newgloops != new_loops.size())
                std::cout << newgloops << "!=" << new_loops.size() << std::endl;
            }
            else tt++;
          }
          else {
            std::cout << "error retrieving loop" << std::endl;
            std::cout << (*mg_it).first << std::endl;
            exit(-1);
          }

          comp++;
       	  if ((comp % 10000000) == 0) {
            oldgloops = newgloops - oldgloops;
            cout << "\t" << comp/10000000 << "0e6 combinations " << newgloops << "/" << oldgloops << " new ";
            cout << tt << "/" << m_sloops.size()*old_loops.size() << std::endl;
            oldgloops = newgloops;
          }
        }
      }
    }
    else std::cout << "too many combinations..." << std::endl;
    cout << tt << "/" << m_sloops.size()*old_loops.size() << endl;

    newbloops = m_bloops.size() - oldbloops;
    cout << "\t\tnew loops " << newgloops << " (" << newbloops << " disconnected) ";
    cout << tt << "/" << m_sloops.size()*old_loops.size() << endl;
    oldbloops = m_bloops.size();

    end = (new_loops.size() == 0) || ((new_loops.size() * m_sloops.size()) >= 10000000);
  } while (!end);
}

void LoopGraph::outputVectorFile(char *str, VectorUint &v1, const VectorUint &v2)
{
  std::ofstream fout(std::string(m_filename + std::string(str)).c_str());
  for (uint i=0; i< v1.size(); i++) {
    fout << i << "\t" << v2[i] << "\t" << v2[i]-v1[i] << std::endl;
    v1[i] = v2[i];
  }
  fout.close();
}

// important: terms are added (i.e. terms can be nonempty)
Real LoopGraph::loopTerm(const VectorLVertex &vloop,
		                     const SetEdges &sedges,
												 VectorTerm &terms,
												 bool get_terms)
{
  Real value = 1.;
  for (VectorLVertexCIt it = vloop.begin(); it != vloop.end(); it++) {

    // get the vertex in the original graph
    Vertex v0 = m_core2g[*it];
    if (gcore[*it].type == VAR) {

      // compute term of the variable
      Real m = (*g)[v0].bel[0] - (*g)[v0].bel[1];  // magnetization
      int q = 0;
      OutEdgeIt ib, ie;
      for (tie(ib, ie) = out_edges(*it, gcore); ib != ie; ib++)
				if (sedges.find(gcore[*ib].label) != sedges.end())
					q++;

      Real a = 1., b = 1., c = 1., m2 = m*m;
      for (int i = 0; i < (q-1); i++) {
        a *= (1. - m);
        b *= (-1. -m);
        c *= (1. - m2);
      }

      // test if c == 0
      b *= -1.;
      Real mu_i = (a + b)/(2.*c);
      if (isinf(mu_i)) { mu_i = 0; }
      value *= mu_i;
      if (get_terms) terms.push_back(Term(*it, mu_i, q));
    }
    else {

      // compute term of the factor
      Real m_j;
      PIndex *index = &(gcore)[*it].index;
      VProb mult((*g)[v0].bel); //prob);

      // for each neighbor j of the factor
			OutEdgeIt j_begin, j_end;
      int q = 0;
      for (tie(j_begin, j_end) = out_edges(*it, gcore); j_begin != j_end; j_begin++) {
				if (sedges.find(gcore[*j_begin].label) != sedges.end()) {
					q++;
          Vertex j = target(*j_begin, gcore);               // variable j
  
          // get the vertex in the original graph
          Vertex vj0 = m_core2g[j];
          m_j = (*g)[vj0].bel[0] - (*g)[vj0].bel[1];//(*g)[vj0].m; //bel[0] - (*g)[vj0].bel[1];
          
          // foreach value of the factor
          for (uint x = 0; x < mult.size(); x++) { 
            mult[x] *= (index->get(x, gcore[*j_begin].pos_index) == 0) ? (+1. - m_j) : (-1. - m_j);
          }
				}
      }
      Real mu_a = 0.;
      for (uint i = 0; i < mult.size(); i++) {
				mu_a += mult[i];
			}
      if ( isinf(mu_a) ) {
       	throw "factor term is infinity";
      }
      value *= mu_a;
      if (get_terms) terms.push_back(Term(*it, mu_a, q));
    }    
  }
  if (isnan(value)) {
    value = 0.;
  }
  return value;
}

Loop LoopGraph::gLoopTerm(MapGLoopIt &it, bool get_terms)
{
  VectorTerm terms;

  Real term = 1.;
  if ((*it).second.vloop.size() == 1) {

    // Disconnected loops: multiply vertices of all the components
    std::vector<MapGLoopIt> itg = m_bloops[(*it).first];
    for (uint l = 0; l < itg.size(); l++) {

      // terms are added sequentially in loopTerm
      term *= loopTerm((*itg[l]).second.vloop, (*itg[l]).second.sedges, terms, get_terms);
    }
  }
  else {
    term = loopTerm((*it).second.vloop, (*it).second.sedges, terms, get_terms);
  }

  Loop ret(term, it, terms); 
  return ret;
}

Real LoopGraph::computeLoopTerms(MapGLoop &loops, bool get_terms)
{
  Real sum_loops = 0.;
  for (MapGLoopIt it = loops.begin(); it != loops.end(); it++) {
    VectorTerm terms;
    Loop loop = gLoopTerm(it, get_terms);// term = gLoopTerm(it, terms);
    m_loops.push_back(loop);
    sum_loops += loop.R;
  }
  return sum_loops;
}
 
// given a vector of (x_i-m_i) mu, termsDeriv returns a vector mur such that mur_i = \prod_{j\neqi} (x_j-m_j)
Real LoopGraph::factorsDeriv(Real pleft, uint idx, std::vector<Real> &mu, std::vector<Real> &mur)
{
  Real pright;
  if (idx == (mu.size()-1)) {

    // base case
    pright = mu[idx];
    mur[idx] = pleft;
  }
  else {

    // recursive case
    pright = factorsDeriv(pleft*mu[idx], idx+1, mu, mur);
    mur[idx] = pleft*pright;
    pright *= mu[idx];
  }
  return pright;
}

// given a vector of terms mu, termsDeriv returns a vector mur such that mur_i = \prod_{j\neqi} mu_j
Real LoopGraph::termsDeriv(Real pleft, uint idx, VectorTerm mu, std::vector<Real> &mur)
{
  Real pright;
  if (idx == (mu.size()-1)) {
    pright = mu[idx].mu;
    mur[idx] = pleft;
  }
  else {

    // recursive case
    pright = termsDeriv(pleft*mu[idx].mu, idx+1, mu, mur);
    mur[idx] = pleft*pright;
    pright *= mu[idx].mu;
  }
  return pright;
}

// just works for the binary case
bool LoopGraph::doLoopMarginalsClamp(int incr_loops)
{
  // resize properly the tlsbp beliefs, the Z sum, and the error vector according to incr_loops
  // if incr_loops > 0 then a matrix of dims [nloops][2]
  // else we only store the marginals considering all found loops
  if (incr_loops) {
    for (uint var = 0; var < m_vars_labels.size(); var++) 
      (*g)[m_vars_labels[var]].bel_tlsbp.resize(m_loops.size(), 2);    //binary case
    m_error_tlsbp.resize(m_loops.size(), -1.);
  }
  else {
    for (uint var = 0; var < m_vars_labels.size(); var++) 
     (*g)[m_vars_labels[var]].bel_tlsbp.resize(1, 2); //binary case
    m_error_tlsbp.resize(1, -1.);
  }	

  clock_t init_clock, end_clock;
  init_clock = ttoc();

  Real err_c0 = -1., err_c1 = -1.;
  Real max_err2clamp = -1.;
  
  // traverse all the variables of the original factor graph
	// only compute those which have a positive label
	// (negative are considerede dummies)

  int nodummies = 0;
	for (uint var = 0; var < m_vars_labels.size(); var++)
		if ((*g)[m_vars_labels[var]].label >= 0)
			nodummies++;

	std::ofstream fbel;
	if ( m_output_tlsbp ) {
		char sfbel[100];
		sprintf(sfbel, "%s.tlsbp.bel", m_filename.c_str());
		fbel.open(sfbel);
    if (!fbel.good())
		  throw "... cannot open the file " + std::string(sfbel);
	}

	int nclamped = 0;
	for (uint var = 0; var < m_vars_labels.size(); var++) {

		if ((*g)[m_vars_labels[var]].label >= 0) {
      bool conv_clamp = true;
			std::cout << "clamping var " << (*g)[m_vars_labels[var]].label;
			std::cout << " " << nclamped << "/" << nodummies << std::endl;
  
      // clamp the var
      Real Ztotal = 0;
      VProb bel(2, 0.);
      
      for (int value1 = 0; value1 < 2; value1++) {
        int value2 = (!value1) ? 1 : 0;
        std::map<Vertex, VProb> psis = clamp(var, value1);
  
        int tries = 0;
        do {
          setTBPVariant((TBPVariant)m_bp_variant);
          reset();
          doBP(m_tol, m_max_iter);
          conv_clamp = ( m_maxdif < m_tol );
          if (!conv_clamp) {
            cout << "NO CONVERGENCE USING " << endl;
            switch (m_bp_variant) {
              case BP_SEQFIX: cout << "\tBP_SEQFIX" << endl; break;
              case BP_SEQRND: cout << "\tBP_SEQRND" << endl; break;
              case BP_MAXRES: cout << "\tBP_MAXRES" << endl; break;
              case BP_PARAL: cout << "\tBP_PARAL" << endl; break;
            }
            m_bp_variant = (m_bp_variant+1) % 4;
            tries += 1;
          }
          else {
  
            // Compute all the loop terms again...
            Real acum = 0., Zacum = 0.;
            for (uint nloop = 0; nloop < m_loops.size(); nloop++) {
              Loop loop = gLoopTerm(m_loops[nloop].it, false);
              acum += loop.R;
              if (isinf(Zacum)) {
                cout << "Z inf. log = " << m_logZ_bp + logl(1. + acum) << endl;
                exit(-1);
              }
              if (incr_loops) {
                Zacum = expl( (long double)(m_logZ_bp + logl(1. + acum)) );
                (*g)[m_vars_labels[var]].bel_tlsbp[nloop][value1] = Zacum;
              }
            }
            if (!incr_loops) {
              Zacum = expl( (long double)(m_logZ_bp + logl(1. + acum)) );
              (*g)[m_vars_labels[var]].bel_tlsbp[0][value1] = Zacum;
            }
          } // end if convergence
        } while ( (!conv_clamp) && (tries < 5) );
  
        // check convergence for a clamped variable
        if (!conv_clamp) return false;
        
        // unclamp the var      
        unclamp(var, psis);
    
        if (incr_loops) {
          for (uint nloop = 0; nloop < m_loops.size(); nloop++) {
            Real Zsum = expl( m_logZ_tlsbp );
  					(*g)[m_vars_labels[var]].bel_tlsbp[nloop][value1] /= Zsum;
            (*g)[m_vars_labels[var]].bel_tlsbp[nloop][value2] = 1 - (*g)[m_vars_labels[var]].bel_tlsbp[nloop][value1];
  
            // should be normalized but ...
  					(*g)[m_vars_labels[var]].bel_tlsbp[nloop].normalizeL1();
  
  					Real err = fabsl( (*g)[m_vars_labels[var]].p[value1] - (*g)[m_vars_labels[var]].bel_tlsbp[nloop][value1] );
            if (err > m_error_tlsbp[nloop]) m_error_tlsbp[nloop] = err;
            if (!value1 && (err > err_c0)) err_c0 = err;
            else if (value1 && (err > err_c1)) err_c1 = err;
  
            err = fabsl( (*g)[m_vars_labels[var]].p[value2] - (*g)[m_vars_labels[var]].bel_tlsbp[nloop][value2] );
            if (err > m_error_tlsbp[nloop]) m_error_tlsbp[nloop] = err;
            if (!value1 && (err > err_c0)) err_c0 = err;
  					else if (value1 && (err > err_c1)) err_c1 = err;
  				}
        }
        else {
  				Ztotal += (*g)[m_vars_labels[var]].bel_tlsbp[0][value1];
  				bel[value1] = (*g)[m_vars_labels[var]].bel_tlsbp[0][value1];
          Real Zsum = expl( m_logZ_tlsbp );
          (*g)[m_vars_labels[var]].bel_tlsbp[0][value1] /= Zsum;
          (*g)[m_vars_labels[var]].bel_tlsbp[0][value2] = 1 - (*g)[m_vars_labels[var]].bel_tlsbp[0][value1];
  
          // should be normalized but ...
          (*g)[m_vars_labels[var]].bel_tlsbp[0].normalizeL1();
  
          /*	
  	cout << "p[" << value1 << "] = " << (*g)[m_vars_labels[var]].p[value1] << endl;
  	cout << "b[" << value1 << "] = " << (*g)[m_vars_labels[var]].bel_tlsbp[0][value1] << endl;
  	cout << "\terr = " << fabsl( (*g)[m_vars_labels[var]].p[value1] - (*g)[m_vars_labels[var]].bel_tlsbp[0][value1] ) << endl;
  	cout << "p[" << value2 << "] = " << (*g)[m_vars_labels[var]].p[value2] << endl;
  	cout << "b[" << value2 << "] = " << (*g)[m_vars_labels[var]].bel_tlsbp[0][value2] << endl;
  	cout << "\terr = " << fabsl( (*g)[m_vars_labels[var]].p[value2] - (*g)[m_vars_labels[var]].bel_tlsbp[0][value2] ) << endl;*/
       
          Real err = fabsl( (*g)[m_vars_labels[var]].p[value1] - (*g)[m_vars_labels[var]].bel_tlsbp[0][value1] );	     
          if (err > m_error_tlsbp[0]) m_error_tlsbp[0] = err; 
  				if (!value1 && (err > err_c0)) err_c0 = err;
  				else if (value1 && (err > err_c1)) err_c1 = err;
  
          err = fabsl( (*g)[m_vars_labels[var]].p[value2] - (*g)[m_vars_labels[var]].bel_tlsbp[0][value2] );
  				if (err > m_error_tlsbp[0]) m_error_tlsbp[0] = err;
  				if (!value1 && (err > err_c0)) err_c0 = err;
  				else if (value1 && (err > err_c1)) err_c1 = err;	
  
        } // end if incr_loops
        
      } // end clamped value
  
      bel[0] /= Ztotal; bel[1] /= Ztotal;
      Real err2clamp = fabsl( (*g)[m_vars_labels[var]].p[0] - bel[0] );
  //    cout << "err b[0] = " << err2clamp << " ";
  
      if (err2clamp > max_err2clamp)
  			max_err2clamp = err2clamp;
      err2clamp = fabsl( (*g)[m_vars_labels[var]].p[1] - bel[1] );
  //    cout << "err b[1] = " << err2clamp << " " << endl;
  
      if (err2clamp > max_err2clamp) max_err2clamp = err2clamp;

			fbel << (*g)[m_vars_labels[var]].label << bel[0] << "\t" << bel[1] << std::endl;

			nclamped++;
		} // end dummy var
  } // end forall vars

	if (m_output_tlsbp)	fbel.close();

  // end cpu-clock
  end_clock = ttoc();
  m_cpu_lcm = end_clock - init_clock;

  std::cout << endl << "Spent " << end_clock - init_clock << " time units (";
  std::cout << toSeconds(end_clock - init_clock) << " scnds) computing marginals" << endl;

  m_error_tlsbp[0] = (err_c0 > err_c1) ? err_c1 : err_c0;
  m_error_tlsbp[0] = (m_error_tlsbp[0] > max_err2clamp) ? max_err2clamp : m_error_tlsbp[0];

  return true;
}

bool LoopGraph::doLoopMarginalsLR(int incr_loops)
{
  m_error_tlsbp.resize(m_loops.size(), -1.);

  // do Linear Response
  doLR(m_tol, m_max_iter);

  OutEdgeIt j_begin, j_end, i_begin, i_end;
  bool foundi, foundj;
 
  Real Zq = expl((long double)(m_logZ_bp - m_logZ_tlsbp));        // second term Z_{BP}/Z_{TLSBP}

  // compute the marginals using Linear response results
  // forall var
  for (VectorVertexIt itv = m_vars_labels.begin(); itv != m_vars_labels.end(); itv++) {
    Vertex k = *itv;                                                 // the variable
    (*g)[k].bel_tlsbp.resize(m_loops.size(), (*g)[k].bel.size());    // a matrix of dims [nloops][2]
    VProb b_tlsbp((*g)[k].bel);                                      // the TLSBP belief

    // forall values of k
    for (int yk = 0; yk < 2; yk++) {
//cout << "value " << yk << endl;

      // forall loops
      Real sum_partial_loops = 0.;
      for (uint l = 0; l < m_loops.size(); l++) {
//cout << "\tloop " << l << endl;

        MapGLoopIt it = m_loops[l].it;

        // compute the loop term r(C) and obtain all the node terms
        // Real rC =
        Loop loop = gLoopTerm(it, true);
        VectorTerm terms = loop.terms;
 
        //cout << "\tterm R = " << rC << endl;

        // compute the precalc derivatives for all the node terms
        std::vector<Real> termsd(terms.size(), 0.);
        termsDeriv(1., 0, terms, termsd);
        Real derivR = 0.;

        // finally, compute the derivative forall node terms
        for (uint jj = 0; jj < terms.size(); jj++) {

//cout << "\t\tnode " << jj << endl;

          Real deriv;           // \fracpartial {\mu}{\theta_k}
          if ((*g)[m_core2g[terms[jj].vertex]].type == VAR) {

            // variable node
            // covariances
            Vertex vj = terms[jj].vertex;
            Vertex vjg = m_core2g[vj];
            VProb cov = (*g)[vjg].m_cov[m_map_LR_vars[k]];
            Real partialm = (!yk) ? (cov[0]-cov[1]) : (cov[2]-cov[3]);
//		Real partialm = (!yk) ? 2*cov[0] : 2*cov[2];          
            Real mj = (*g)[vjg].m;
            int q = terms[jj].q;
            Real mm = (1-mj), pm = (1+mj), m2 = mj*mj, mm2 = (1-m2), uq = -1.;
            for (int qt = 1; qt < q; qt++) {
              mm *= mm;
              pm *= pm;
              mm2 *= mm2;
              uq *= -1;
            }
            deriv = (1-q) * ( mm + (-1)*uq*pm )/( 2*mm2 ) * partialm;
if (mm2 < 1e-5) { deriv = 0; }
//cout << "\t\t\tmj =  " << mj << " q = " << q << " part = " << partialm << " deriv = " << deriv << endl;
//cout << "\t\t\tmm = " << mm << " pm = " << pm << " mm2 = " << mm2 << endl;
          }
          else {

            // factor node
            Vertex va = terms[jj].vertex;
            Vertex vag = m_core2g[va];
            PIndex *index = &(gcore)[va].index;
            VProb bel((*g)[vag].bel);

            // first term (covariances)
            VProb cov = (*g)[vag].m_cov[m_map_LR_vars[k]], partial(bel.size(), 0.);
            if (!yk) {
              for (uint fi=0; fi<partial.size(); fi++)
                partial[fi] = cov[fi];
            }
            else {
              for (uint fi=0; fi<partial.size(); fi++)
                partial[fi] = cov[partial.size()+fi];
            }

            // foreach value of the factor
            Real deriv2 = 0.;
            for (uint xa = 0; xa < bel.size(); xa++) {

              Real sum1p = 0., mult_cov = partial[xa];

              // foreach neighbor i (variable in the factor) which included in the loop
              for (tie(i_begin, i_end) = out_edges(va, gcore); i_begin != i_end; i_begin++) {

                Real mult = 1.;
                Vertex vi = target(*i_begin, gcore);              // variable 
                Vertex vig = m_core2g[vi];
                foundi = false;
                for (uint i = 0; (i < terms.size()) && !foundi; i++) {     // only variables of the loop
                  if (vi == terms[i].vertex) {
                    foundi = true;

                    // foreach neighbor j (variable in the factor) which included in the loop
                    // again because we need to sum the partial derivatives...
                    for (tie(j_begin, j_end) = out_edges(va, gcore); j_begin != j_end; j_begin++) {

                      Vertex vj = target(*j_begin, gcore);              // variable j
                      Vertex vjg = m_core2g[vj];
                      foundj = false;
                      for (uint j = 0; (j < terms.size()) && !foundj; j++) {     // only variables of the loop
                        if (vj == terms[j].vertex) {
                          foundj = true;

                          // if same variable multiply with derivative
                          if (vj == vi) {
                            VProb cov = (*g)[vjg].m_cov[m_map_LR_vars[k]];
                            Real partialm = (!yk) ? (cov[0]-cov[1]) : (cov[2]-cov[3]);
                            mult *= -partialm;
                          }
                          else {
                            mult *= (index->get(xa, gcore[*j_begin].pos_index) == 0) ? (+1. - (*g)[vjg].m) : (-1. - (*g)[vjg].m);
                          }
                        }
                      }
                      if (!foundj) {cout << "error" << endl; exit(-1);}
                    } // endfor neighbor j

                    sum1p += mult;
                    mult_cov *= (index->get(xa, gcore[*i_begin].pos_index) == 0) ? (+1. - (*g)[vig].m) : (-1. - (*g)[vig].m);
                  }
                }
if (!foundi) {cout << "error" << endl; exit(-1);}
              } // endfor neighbor i
//cout << "\t\t\tfvalue " << xa << " " << bel[xa] << "*" << sum1p << " = " << (bel[xa] * sum1p) << endl;
//cout << "\t\t\t\t+" << mult_cov << " = " << (bel[xa] * sum1p) + mult_cov<< endl;
              deriv2 += (bel[xa] * sum1p) + mult_cov;

            } // endfor all factor values

            deriv = deriv2;

          } // end if type of node

          derivR += (deriv * termsd[jj]);    // sum the node-term
//cout << "deriv = " << deriv << " derivR = " << derivR << endl;
 
        } // end forall nodes terms of the loop

        sum_partial_loops += derivR; // sum the loop-term
/*        if (incr_loops && !yk) {
 
          // save marginals at this point (only for +1 values)
          VProb bel_tlsbp(2, 0.);
          bel_tlsbp[0] = b_tlsbp[yk] + Zq * sum_partial_loops;
          bel_tlsbp[1] = 1 - bel_tlsbp[0];
          (*g)[k].bel_tlsbp.push_back( bel_tlsbp );
        }*/

        (*g)[k].bel_tlsbp[l][yk] = b_tlsbp[yk] + Zq * sum_partial_loops;
        Real err = fabsl( (*g)[k].p[yk] - (*g)[k].bel_tlsbp[l][yk] );
        if (err > m_error_tlsbp[l])
          m_error_tlsbp[l] = err;
 
      } // end forall loops

    } // end forall values of k


cout << "b_tlsbp[" << 0 << "] = " << (*g)[k].bel_tlsbp[m_loops.size()-1][0];
cout << "\t" << "b_tlsbp[" << 1 << "] = " << (*g)[k].bel_tlsbp[m_loops.size()-1][1];
cout << "\tsum = " << (*g)[k].bel_tlsbp[m_loops.size()-1][0]+(*g)[k].bel_tlsbp[m_loops.size()-1][1] << endl;

  } // end forall var k
  return true;
}

void LoopGraph::buildVertex2core()
{
  m_g2core.resize(num_vertices(*g), -1);
  m_core2g.resize(num_vertices(gcore), -1);
  std::pair<VertexIt, VertexIt> itc = vertices(gcore); 

  // forall vertices in the 2-core
  while (itc.first != itc.second) {
    PVertex vc = gcore[*itc.first];
    bool found = false;
    std::pair<VertexIt, VertexIt> itg = vertices(*g);

    // forall vertices in the original graph
    while ( (itg.first != itg.second) && !found ) {
      PVertex vg = (*g)[*itg.first];
      if ( (vg.type == vc.type) && (vg.label == vc.label) ) {
       found = true;
       m_g2core[*itg.first] = *itc.first;
       m_core2g[*itc.first] = *itg.first;
      }
      itg.first++;
    }

    if (!found) cout << "error building vertices correspondences..." << endl;
    itc.first++;
  }
}

bool LoopGraph::doLoopSeries( uint S,
                              uint C,
                              uint B,
                              uint M,
                              int incr_loops, 
                              bool load,
                              bool save,
                              std::string loadfile,
															bool output)
{
  //  m_hash.resize(HASH_CTE, 0);
  // general arguments
  m_C = C;
	m_output_tlsbp = output;

  if (!B) std::cout << "using max_length of the largest simple-loop found" << std::endl;
  else std::cout << "using max_length bound of " << B << std::endl;
  m_max_length = B;

  m_cpu_lc = 0;
  m_cnbloops = 0;
  m_cbloops = 0;
  m_ncbloops = 0;
  m_ncnbloops = 0;

  // check that loops can be saved
  if (save) {
    char filename[250];
    std::string subfilename(m_filename.substr(m_filename.rfind("/")+1));
    sprintf(filename, "%s_loops.lps", subfilename.c_str());
    std::ofstream fout(filename);
    if (fout.is_open()) {
      fout.close();
    }
    else {
      std::cout << "graph file could not be created: " << filename << std::endl;
      std::cout << "make sure you have write permissions" << std::endl;
      exit(-1);
    }
  }

  // obtain the 2-core removing vertices with degree 1
  gcore = *g;
  preprocessGraph();
  if (num_vertices(gcore) > 1) {

    // store the Vertexs id mapping in m_vertices_g0;
    buildVertex2core();  
  
    m_S = S;
  
    // check Vertex type
    checkType();
  
    // set the ordering labels to the edges
    // Note that this changes the labels of the edges, not the variable labels...
    int label = 0;
    EdgeIt ei, ei_end;
    for (tie(ei, ei_end) = edges(gcore); ei != ei_end; ei++, label++) {
      m_edges.push_back(*ei);
      gcore[*ei].label = label;
    }
   
    // number of bytes per edge (also sum the separator)
    m_nbytes = (uint)log10((double)num_edges(gcore)) + 2;
    m_nvertices = num_vertices(gcore);
  
    // start cpu-clock
    clock_t init_clock, end_clock;
    init_clock = ttoc();
  
    if (load) {
  
      std::cout << "loading loops from file " << loadfile << std::endl;
      loadLoops(loadfile);
      std::cout << "Total loaded Loops : " << m_sloops.size() + m_gloops.size() << std::endl;
  
      // compute loop terms
      m_loops.clear();
      computeLoopTerms(m_sloops);
      computeLoopTerms(m_gloops);
    }
    else {
  
      // count simple loops
      countSLoops();
      std::cout << std::endl;
      std::cout << "Total: " << m_sloops.size() << " simple loops found" << std::endl;
      for (uint i=0; i< m_vsizes.size(); i++)
        if (m_vsizes[i]) std::cout << "\tsize " << i << "\t" << m_vsizes[i] << std::endl;
      std::cout << std::endl;
  
      // count the rest of generalized loops
			if ( m_sloops.size() > 1 )
	      countGLoops();
    }   // endif all
  
    // end cpu-clock
    end_clock = ttoc();
  
    std::cout << "Spent " << end_clock - init_clock << " time units (";
    std::cout << toSeconds(end_clock - init_clock) << " scnds). ";
		if ( m_sloops.size() > 0 ) {
			std::cout << "Sorting loops ..."; std::cout.flush();
		}
    m_cpu_lc = end_clock - init_clock;
  
    // final results
    // sort found loops by contributions
    sort(m_loops.begin(), m_loops.end(), LessLoop());
  
    std::cout << "... " << " Total loops = " << m_loops.size() << std::endl;
    std::cout << "s = " << m_sloops.size() << " ";
    std::cout << "g = " << m_gloops.size() << " (";
    std::cout << "cb = " << m_cbloops << " ";
    std::cout << "cnb = " << m_cnbloops << " ";
    std::cout << "ncb = " << m_ncbloops << " ";
    std::cout << "ncnb  = " << m_ncnbloops << ")" << std::endl;
  
    m_vlogZ_tlsbp.clear();
    Real acum = 0.;
    std::ofstream fcontrib("contrib.txt");
    m_error_Ztlsbp.resize(m_loops.size(), -1);
  
    Real Z_tlsbp = 0;
    for (uint lc = 0; lc < m_loops.size(); lc++) {
      acum += m_loops[lc].R;
      m_logZ_tlsbp = m_logZ_bp + logl(1. + acum);
      if ( isnan(m_logZ_tlsbp) ) {
  //      cout << "loop " << lc << "\tisnan Z , acum = " << acum << endl;
      }
      m_vlogZ_tlsbp.push_back( m_logZ_tlsbp );
      fcontrib << setprecision(15) << lc << " " << m_loops[lc].R << " " << getSizeLoopEdges(m_loops[lc].it) << std::endl;
      m_error_Ztlsbp[lc] = fabsl(m_logZ_exact - m_logZ_tlsbp);
      //m_error_Ztlsbp[lc] = fabsl(expl(m_logZ_tlsbp - m_logZ_exact) - 1);
    }
  
		std::ofstream fcum("cum_err.txt");
		for (uint lc = 0; lc < m_error_Ztlsbp.size(); lc++)
			fcum << setprecision(15) << m_error_Ztlsbp[lc] << std::endl;
		fcum.close();

    Z_tlsbp = expl(m_logZ_bp)*(1. + acum);
    
    fcontrib.close();
    std::cout << std::endl;
    std::cout << m_loops.size() << " loops considered ";
  
    m_error_bp = getBPError();
    Real bethe = m_logZ_bp;
  
#ifdef DOT_LOOPS
  
    // output loops  
    if ( m_loops.size() > 25 ) std::cout << "dot files won't be generated (nfactors > 25)..." << std::endl;
    else {
      for (uint nloop = 0; nloop < m_loops.size(); nloop++) {
        Loop out_loop = m_loops[nloop];
        char b[80];
        sprintf(b, "out_loop%d.dot", nloop);
        std::cout << "Loop " << nloop << std::endl;
        loop2Dot( out_loop.it, b );
      }
    }
#endif
  
    // compute marginals
    Real cout_error_marginals = -1;
    switch (M) {
    case 1:
      // Compute single-node marginals using the clamping method
      if (!doLoopMarginalsClamp(incr_loops))
        return false;
  
      // keep the old value of the Bethe Free energy
      m_logZ_bp = bethe;
      cout_error_marginals = (incr_loops) ? m_error_tlsbp[m_loops.size()-1] : m_error_tlsbp[0];
     break;
    case 2:
  
      // Compute single-node marginals using Linear Response
      // copy the original graph to a new graph 
      doLoopMarginalsLR(incr_loops);
      cout_error_marginals = (incr_loops) ? m_error_tlsbp[m_loops.size()-1] : m_error_tlsbp[0];
      break;
    default:
  
      // Do not compute single-node marginals
      cout << "Not computing single-node marginals..." << endl;
    }

    // output results
	  cout.precision(14);
	  cout << std::endl;
 	  cout << "\tlogZ_exact  = " << m_logZ_exact << " " << expl(m_logZ_exact) << endl;
	  cout << "\tlogZ_tlsbp  = " << m_logZ_tlsbp << " " << expl(m_logZ_tlsbp) <<  endl;
	  cout << "\tlogZ_bp     = " << m_logZ_bp << " " << expl(m_logZ_bp) << endl;
	  cout << endl;
	  cout << "\terr_Z_bp    = " << fabsl(m_logZ_bp - m_logZ_exact) << endl;
	  cout << "\terr_b_bp    = " << m_error_bp << endl;
	  cout << endl;
	  cout << "\terr_Z_tlsbp = " << fabsl(m_logZ_tlsbp - m_logZ_exact) << endl;
	  cout << "\terr_b_tlsbp = " << cout_error_marginals << endl;
	  cout << endl;
  }
	else { 
		std::cout << "BP is exact" << std::endl;
	  cout << "\tlogZ_exact  = " << m_logZ_exact << " " << expl(m_logZ_exact) << endl;
	  cout << "\tlogZ_bp     = " << m_logZ_bp << " " << expl(m_logZ_bp) << endl;
	  cout << endl;
	  cout << "\terr_Z_bp    = " << fabsl(m_logZ_bp - m_logZ_exact) << endl;
	  cout << "\terr_b_bp    = " << m_error_bp << endl;
  }


	// for the moment, output errors of BP 
  std::ofstream fout;
	if ( m_output_tlsbp ) {
		char sfout[100];
		sprintf(sfout, "%s.err", m_filename.substr(m_filename.find_last_of('/')+1, m_filename.size()-m_filename.find_last_of('/')).c_str());
		std::cout << "Opening file " << sfout << " for writing" << std::endl;
		fout.open( sfout );
		if (!fout.good())
			throw "... cannot open the file " + std::string(sfout);

		// Format:   [P_EXACT(x=+1), P_BP(x=+1), ERROR]
		// or just:  [P_BP(x=+1)]
		fout << setprecision(14);
    for (VectorVertexIt itv = m_vars_labels.begin(); itv != m_vars_labels.end(); itv++) {
			if ( ((*g)[*itv].label >= 0) ) {
				if ( m_exact ) {
          fout << (*g)[*itv].label << "\t" << (*g)[*itv].p[1] << "\t" << (*g)[*itv].bel[1];
					double err0 = fabs( (*g)[*itv].bel[0] - (*g)[*itv].p[0] );
					double err1 = fabs( (*g)[*itv].bel[1] - (*g)[*itv].p[1] );
					double err = (err0 < err1) ? err0 : err1;
     		  fout << "\t" << err << std::endl;
	  		}
				else {
					fout << (*g)[*itv].label << "\t" << (*g)[*itv].bel[1] << std::endl;
				}
      }    
    }
	}
	fout.close();

  // save loops if required
  if (save) {
    char filename[250];
    std::string subfilename(m_filename.substr(m_filename.rfind("/")+1));
    sprintf(filename, "%s_loops.lps", subfilename.c_str());
    std::string loops_file(filename);
    std::cout << "saving loops to file " << loops_file << std::endl;
    saveLoops(loops_file);
  }
  return true;
}

void LoopGraph::saveLoops(std::string filename)
{
  // vector of simple loops sizes
  std::ofstream fout(filename.c_str());

  // m_smaxsize
  fout << "max_length" << std::endl;
  fout << m_max_length << std::endl;

  // m_steps
  fout << "steps" << std::endl;
  fout << m_steps << std::endl;

  fout << "vsizes" << std::endl;
  for (uint i = 0; i < m_vsizes.size(); i++) fout << m_vsizes[i] << " ";
  fout << std::endl;

  // simple loops
  fout << "simple" << std::endl;
  for (MapGLoopIt it = m_sloops.begin(); it != m_sloops.end(); it++) {
    fout << (*it).first << " ";
    for (uint i=0; i<(*it).second.vloop.size(); i++) fout << (*it).second.vloop[i] << " ";
    fout << std::endl;
    if (((*it).first.size()+(*it).second.vloop.size()*3)> MAX_LOOP) {
      std::cout << "Found key with length " << (*it).first.size() << std::endl;
      std::cout << "Increase MAX_LOOP" << std::endl;
    }
  }
  fout << "endsimple" << std::endl;

  // vector of general loops sizes vertices
  fout << "vgsizes_vertices" << std::endl;
  for (uint i = 0; i < m_vgsizes_vertices.size(); i++) fout << m_vgsizes_vertices[i] << " ";
  fout << std::endl;

  // vector of general loops sizes edges
  fout << "vgsizes_edges" << std::endl;
  for (uint i = 0; i < m_vgsizes_edges.size(); i++) fout << m_vgsizes_edges[i] << " ";
  fout << std::endl;

  // vector of disconnected loops sizes vertices
  fout << "vbsizes_vertices" << std::endl;
  for (uint i = 0; i < m_vbsizes_vertices.size(); i++) fout << m_vbsizes_vertices[i] << " ";
  fout << std::endl;

  // vector of disconnected loops sizes edges
  fout << "vbsizes_edges" << std::endl;
  for (uint i = 0; i < m_vbsizes_edges.size(); i++) fout << m_vbsizes_edges[i] << " ";
  fout << std::endl;

  // general non-disconnected loops
  fout << "gen" << std::endl;
  for (MapGLoopIt it = m_gloops.begin(); it != m_gloops.end(); it++) {
    if ((*it).second.vloop.size() > 1) {

      fout << (*it).first << " ";
      for (uint i=0; i<(*it).second.vloop.size(); i++) fout << (*it).second.vloop[i] << " ";
      fout << std::endl;

      if (((*it).first.size()+(*it).second.vloop.size()*2)> MAX_LOOP) {
	std::cout << "Found key with length " << (*it).first.size() << std::endl;
	std::cout << "Increase MAX_LOOP" << std::endl;
	////exit(-1);
      }
    }
  }
  fout << "end" << std::endl;

  // general loops non-disconnected loops
  fout << "br" << std::endl;
  for (MapBLoopIt it = m_bloops.begin(); it != m_bloops.end(); it++) {
    fout << (*it).first << " " << (*it).second.size() << std::endl;
    for (uint c = 0; c < (*it).second.size(); c++) {
      if ((*it).second.size() > 1) {
        fout << (*(*it).second[c]).first << std::endl;;
      }
      else {
        std::cout << "error size of bloop < 2!" << std::endl;
        throw "Bad format loops file";
      }
    }
  }
  fout << "end" << std::endl;

  fout.close();
}


bool LoopGraph::loadVector(std::string vname, VectorUint &v, std::ifstream &fin)
{
  std::string strbuf;
  strbuf = getLineFile(fin);
  if (strbuf != vname) {
    std::cout << "Syntax error loading loops " << vname << std::endl;
    //exit(-1);
  }

  v.clear();
  strbuf = getLineFile(fin);
  uint poso = 0, posv;
  do {
    posv = strbuf.find(" ", poso);
    v.push_back(atoi(strbuf.substr(poso, posv-poso).c_str()));
    poso = posv+1;
  }
  while (poso < strbuf.length());
  return true;
}

bool LoopGraph::loadLoops(std::string filename)
{
  // simple loops
  std::ifstream fin(filename.c_str());
  if (!fin) {
    std::cout << "Loops file " << filename << " not found!" << std::endl;
    //exit(-1);
  }

  std::string strbuf;
  strbuf = getLineFile(fin);
  if (strbuf != "max_length") {
    std::cout << "Syntax error loading loops max_length" << std::endl;
    throw "Bad format loops file";
  }
  fin >> m_max_length;
  strbuf = getLineFile(fin);
  std::cout << m_max_length << std::endl;
  
  strbuf = getLineFile(fin);
  if (strbuf != "steps") {
    std::cout << "Syntax error loading loops steps" << std::endl;
    throw "Bad format loops file";
  }
  fin >> m_steps;

  strbuf = getLineFile(fin);
  std::cout << m_steps << std::endl;
  
  loadVector("vsizes", m_vsizes, fin);
  
  strbuf = getLineFile(fin);
  if (strbuf != "simple") {
    std::cout << "Syntax error loading loops" << std::endl;
    throw "Bad format loops file";
  }
  std::cout << "Loading loops ... " << std::endl;
  strbuf = getLineFile(fin);
  do {
    
    // process loop
    int pos = strbuf.rfind("-");
    std::string key(strbuf, 0, pos+1);
    VectorLVertex loopv;
    uint poso = pos+2, posv;
    do {
      posv = strbuf.find(" ", poso);
      loopv.push_back(atoi(strbuf.substr(poso, posv-poso).c_str()));
      poso = posv+1;
    }
    while (poso < strbuf.length());
    MapLoopElem newloop(loopv, getEdges(key), getEdges(key).size() );
    m_sloops[key] = newloop;
    
    // new loop
    strbuf = getLineFile(fin);
  } while (strbuf != "endsimple");
  
  std::cout << "\t" << m_sloops.size() << " simple loops " << std::endl;
  
  loadVector("vgsizes_vertices", m_vgsizes_vertices, fin);
  loadVector("vgsizes_edges", m_vgsizes_edges, fin);
  loadVector("vbsizes_vertices", m_vbsizes_vertices, fin);
  loadVector("vbsizes_edges", m_vbsizes_edges, fin);
  
  strbuf = getLineFile(fin);
  if (strbuf != "gen") {
    std::cout << "Syntax error loading loops" << std::endl;
    throw "Bad format loops file";
  }
  strbuf = getLineFile(fin);
  while (strbuf != "end") {

    // process loop
    int pos = strbuf.rfind("-");
    std::string key(strbuf, 0, pos+1);
    VectorLVertex loopv;
    uint poso = pos+2, posv;
    do {
      posv = strbuf.find(" ", poso);
      loopv.push_back(atoi(strbuf.substr(poso, posv-poso).c_str()));

      poso = posv+1;
    }
    while (poso < strbuf.length());
    MapLoopElem newloop(loopv, getEdges(key), getEdges(key).size() );
    m_gloops[key] = newloop;
    
    // new loop
    strbuf = getLineFile(fin);
  }
  std::cout << "\t" << m_gloops.size() << " non disconnected generalized loops " << std::endl;
  
  strbuf = getLineFile(fin);
  if (strbuf != "br") {
    std::cout << "Syntax error loading loops" << std::endl;
    throw "Bad format loops file";
  }
  strbuf = getLineFile(fin);
  while (strbuf != "end") {

    // process loop
    int pos = strbuf.rfind("-");
    std::string key_total(strbuf, 0, pos+1);
    VectorLVertex dmmy(1,0);
    MapLoopElem loop_dmmy(dmmy, SetEdges(), 0);
    m_gloops[key_total] = loop_dmmy;

    int ncomp = atoi(strbuf.substr(pos+1).c_str());
    std::vector<MapGLoopIt> vcomps;
    for (int c = 0; c < ncomp; c++) {
      strbuf = getLineFile(fin);
      std::string key_comp(strbuf, 0, pos+1);

      MapGLoopIt loopit;
      MapGLoopIt itsloop = m_sloops.find(key_comp);
      MapGLoopIt itgloop = m_gloops.find(key_comp);
      if ( (itsloop != m_sloops.end()) && (itgloop == m_gloops.end()) ) loopit = itsloop;
      else if ( (itsloop == m_sloops.end()) && (itgloop != m_gloops.end()) ) loopit = itgloop;
      else if ( (itsloop != m_sloops.end()) && (itgloop != m_gloops.end()) ) {
        std::cout << "Error loading disconnected loops component found in both loop structs" << std::endl;
        throw "Bad format loops file";
      }
      else if ( (itsloop == m_sloops.end()) && (itgloop == m_gloops.end()) ) {
        std::cout << "Error loading disconnected loops component " << key_comp << " not found in both structs" << std::endl;
        throw "Bad format loops file";
      }
      vcomps.push_back(loopit);
    }
    m_bloops[key_total] = vcomps;
    
    // new loop
    strbuf = getLineFile(fin);
  }
  std::cout << "\t" << m_gloops.size() << " generalized loops " << std::endl;

  return true;
}

