#ifndef __Stat
#define __Stat

#include <math.h>
#include "global.hh"

/// The Stat class contains the required methods for computing statistics of a sampling distribution.

/**
   Basically computes efficiently the mean and the stdev of a distribution which is obtained in realtime.
 */
class Stat {

public:

  int size;              ///< Size of the sample
  Real mean;           ///< Present Mean of the sequence
  Real mean2;          ///< Present Mean of the sequence ^2
  Real max;            ///< Maximum of the sequence
  Real min;            ///< Minimum of the sequence

  /// Default constructor
  /** details
   */
  Stat() { mean  = 0.0; mean2 = 0.0; size = 0; max = -1e20; min = +1e20; }

  /// Destructor
  /** details
   */
  virtual ~Stat() {}

  /// Resets the sampling.
  /** details
   */
  void resetStats() { mean  = 0.0; mean2 = 0.0; size = 0; max = -1e20; min = +1e20; }

  /// Adds one sample data.
  /** Updates the mean and the squared mean for obtaining the moments in an efficient way.
   */
  void addSample(Real elem) {
    size++;
    mean  = mean  + ((Real)elem - mean) / ((Real)size);
    mean2 = mean2 + ((Real)(elem*elem) - mean2) / ((Real)size);
    min = (elem < min) ? elem : min;
    max = (elem > max) ? elem : max;
  }

  /// Returns the first moment of the sampling distribution : \f$\displaystyle{E[x]}\f$
  /** details
   */
  Real getMean() { return mean; }

  /// Returns the Variance of the sampling distribution: \f$\displaystyle{E[x^2] - {E[x]}^2}\f$
  /** details
   */
  Real getVar()  { return mean2 - mean*mean; }

  /// Returns the Standard deviation of the sampling distribution: \f$\sqrt{\displaystyle{E[x^2] - {E[x]}^2}}\f$
  /** details
   */
  Real getStdev()  { Real s = sqrt(mean2 - mean*mean); return (s!=s) ? 0.0 : s; }

  /// Returns the Maximum of the sampling distribution:
  /** details
   */
  Real getMax()  { return max; }

  /// Returns the Minimum of the sampling distribution:
  /** details
   */
  Real getMin()  { return min; }
};

#endif


