#include "pindex.hh"
#include "stat.hh"
#include "loopgraph.hh"

#include <iostream>
#include <vector>
#include <algorithm>
#include <fstream>
#include <iomanip>
#include "testai.hh"
 
int main(int argc, char* argv[])
{
  // TLSBP options are:
  // S       : # simple-loops (required)
  // C       : depth of complex-loops search (required)
  // B       : max_length bound
  // load    : load loops
  // save    : save loops
  // M       : computation of marginals
  // I       : incremental loops
  
  try {
    string filename;
    string aliases;
    vector<string> methods;
    double tol;
    size_t maxiter;
    size_t verbose;
 
    int S, C, B = 0, M = 1, D = 0;
    bool load = false, save, I;
    string loadloops_filename;

    po::options_description opts_required("Required options");
    opts_required.add_options()
      ("filename", po::value< string >(&filename), "Filename of FactorGraph")
      ("methods", po::value< vector<string> >(&methods)->multitoken(), "AI methods to test")
      ("S", po::value< int >(&S), "# simple-loops")
      ("C", po::value< int >(&C), "depth of complex-loops search")
      ;

    po::options_description opts_optional("Allowed options");
    opts_optional.add_options()
      ("help", "produce help message")
      ("load", po::value< string >(&loadloops_filename), "load loops (filename)")
      ("save", po::value< bool >(&save), "save loops (0 no save, 1 save)")
      ("B", po::value< int >(&B), "max_length bound")
      ("M", po::value< int >(&M), "marginals computation (0 no marginals, 1 clamping method, 2 LR method)")
      ("I", po::value< bool >(&I), "incremental loops")
      ("D", po::value< int >(&D), "marginals corresponding to variables labeled with V > L in the factor graph will not be computed")
      ("aliases", po::value< string >(&aliases), "Filename for aliases")
      ("tol", po::value< double >(&tol), "Override tolerance")
      ("maxiter", po::value< size_t >(&maxiter), "Override maximum number of iterations")
      ("verbose", po::value< size_t >(&verbose), "Override verbosity")
      ;

    po::options_description cmdline_options;
    cmdline_options.add(opts_required).add(opts_optional);

    po::variables_map vm;
    po::store(po::parse_command_line(argc, argv, cmdline_options), vm);      	    
    po::notify(vm);

    if( vm.count("help") || !(vm.count("filename") && vm.count("methods")) ) {
      cout << "Reads factorgraph <filename.fg> and performs the approximate" << endl;
      cout << "inference algorithms <method*>, reporting clocks, max and average" << endl;
      cout << "error and relative logZ error (comparing with the results of" << endl;
      cout << "<method0>, the base method, for which one can use JTREE_HUGIN)." << endl << endl;
      cout << opts_required << opts_optional << endl;
      return 1;
    }

    if (vm.count("load"))
      load = true;
    Real bptol = tol;
    uint bpmaxiter = 100000;
    bool conv = false;
    int variant = 2, tries = 0;

    // Create loop graph
		std::cout << filename << std::endl;
    LoopGraph graph(filename, (TBPVariant)variant);

/*		// remove this:
		vector<int> vclamped;
		vclamped.push_back(5139);
		for (uint vc = 0; vc < vclamped.size(); vc++) {
			cout << "Clamping var " << vclamped[vc] << endl;
			//graph.clampLabel(vclamped[vc], 1);
		}*/

    // Run BP (try the following schedulings)
    while (!conv && (tries < 5) ) {

      switch (variant) {
        case 1: cout << "BP_SEQFIXED" << endl; break;  
        case 2: cout << "BP_SEQRANDOM" << endl; break;  
        case 3: cout << "BP_SEQMAXRES" << endl; break;  
        case 4: cout << "BP_PAR" << endl; break;  
      }

      graph.setTBPVariant( (TBPVariant) variant );

	    // set verbosity to 1 and store some output
      graph.doBP(bptol, bpmaxiter, 1, true);
      if (graph.m_maxdif < bptol) {
        conv = true;
			}
      else {
        cout << graph.m_maxdif << endl;
        variant = (variant+1)%4;
        tries++;
      }
    }

    if (conv) {

      // Read aliases
      map<string,string> Aliases;
      if( !aliases.empty() ) {
        ifstream infile;
        infile.open (aliases.c_str());
        if (infile.is_open()) {
          while( true ) {
            string line;
            getline(infile,line);
            if( infile.fail() )
              break;
            if( (!line.empty()) && (line[0] != '#') ) {
              string::size_type pos = line.find(':',0);
              if( pos == string::npos )
                throw "Invalid alias";
              else {
                string::size_type posl = line.substr(0, pos).find_last_not_of(" \t");
                string key = line.substr(0, posl + 1);
                string::size_type posr = line.substr(pos + 1, line.length()).find_first_not_of(" \t");
                string val = line.substr(pos + 1 + posr, line.length());
                Aliases[key] = val;
              }
            }
          }
          infile.close();
        } else
          throw "Error opening aliases file";
      } 

      FactorGraph fg;
      if( fg.ReadFromFile(filename.c_str()) ) {
        cout << "Error reading " << filename << endl;
        return 2;
      } else {         

/*				// Clamp also in Joris' graph
				for (uint vc = 0; vc < vclamped.size(); vc++) {
					cout << "Clamping var " << vclamped[vc] << endl;
  				int _nn = 0, f = 0;
  				while (!f) {
  					if (fg.var(_nn).label() == vclamped[vc]) {
  						//fg.clamp(fg.var(_nn), 1);
  						cout << "clamped" << endl;f=1;
  					}
  					_nn++;
					}
					if (!f) throw "Error: var not found";
				}*/

        vector<Factor> q0;
        double logZ0 = 0.0;

        cout << "# " << filename << endl;          
        cout.width( 40 );
        cout << left << "# METHOD" << "  ";
        cout.width( 14 );
        cout << right << "CLOCKS" << "    ";
        cout.width( 14 );
        cout << "MAX ERROR" << "  ";
        cout.width( 14 );
        cout << "AVG ERROR" << "  ";
        cout.width( 14 );
        cout << "LOGZ ERROR" << "  ";
        cout.width( 14 );
        cout << "MAXDIFF" << endl;

        for( size_t m = 0; m < methods.size(); m++ ) {
          pair<string, Properties> meth = parseMethod( methods[m], Aliases );

          if( vm.count("tol") )
            meth.second.Set("tol",tol);
          if( vm.count("maxiter") )
            meth.second.Set("maxiter",maxiter);
          if( vm.count("verbose") )
            meth.second.Set("verbose",verbose);
          TestAI piet(fg, meth.first, meth.second );
          piet.doAI();

//        Joris considers that the first method is the exact
					if ( m == 0 ) {        

            q0 = piet.q;
            logZ0 = piet.logZ;
					
            if( meth.first == string("JTREE") ) {
							
              // I consider the JTREE as the only exact method
							// set the exact results into the loopgraph object
              for (uint i=0; i<q0.size(); i++) {
                VProb pi(2);
                for (uint v=0; v<q0[i].stateSpace(); v++) {
//                  std::cout << v << " ";
//                  std::cout << q0[i] << std::endl;
                  pi[v] = q0[i][v];
                }
                graph.setExactMarginal(i, pi);
							}
              graph.m_logZ_exact = logZ0;
            }
          }
						
					piet.calcErrs(q0);

					cout.width( 40 );
          cout << left << methods[m] << "  ";
          cout.width( 10 );
          cout << right << piet.time << "    ";		    

          if( meth.first != string("JTREE") ) {
            cout.setf( ios_base::scientific );
            cout.precision( 14 );
            cout.width( 14 ); 
            double me = clipdouble( piet.maxErr(), 1e-15 );
            cout << me << "  ";
            cout.width( 10 );
            double ae = clipdouble( piet.avgErr(), 1e-15 );
            cout << ae << "  ";
            cout.width( 10 );
            double le = clipdouble( fabs(piet.logZ - logZ0), 1e-15 );
            cout << le << "  ";
            cout.width( 10 );                
            double md = clipdouble( piet.maxdiff, 1e-15 );

            if( isnan( me ) )
              md = me;
            if( isnan( ae ) )
              md = ae;
            cout << md << endl;
          } else
            cout << endl;
        }
      }

      // finally, do loop calculus
      graph.doLoopSeries(S, C, B, M, I, load, save, loadloops_filename, true);
    }   
  } catch(const char *e) {
    cerr << "Exception: " << e << endl;
    return 1;
  } catch(exception& e) {
    cerr << "Exception: " << e.what() << endl;
    return 1;
  }
  catch(...) { 
    cerr << "Exception of unknown type!" << endl;
  }

  return 0;
}

