#include <iostream>
#include <fstream>
#include <map>
#include <numeric>
#include <cmath>
#include <cstdlib>

#include "util.h"
#include "alldai.h"

#include <boost/program_options.hpp>

using namespace std;
namespace po = boost::program_options;

class TestAI {
protected:
  InfAlg          *obj;
  string          name;
  vector<double>  err;

public:
  vector<Factor>  q;
  double          logZ;
  double          maxdiff;
  clock_t         time;

  TestAI( const FactorGraph &fg, const string &_name, const Properties &opts ) : obj(NULL), name(_name), err(), q(), logZ(0.0), maxdiff(0.0), time(0)	{
    clock_t tic = toc();
    obj = newInfAlg( name, fg, opts );
    time += toc() - tic;
	}

  ~TestAI() { 
    if( obj != NULL )
        delete obj;
  }

  string identify() { 
    if( obj != NULL )
      return obj->identify(); 
    else
      return "NULL";
  }

  vector<Factor> allBeliefs() {
    vector<Factor> result;
    for( size_t i = 0; i < obj->nrVars(); i++ )
      result.push_back( obj->belief( obj->var(i) ) );
    return result;
  }

  void doAI() {
    clock_t tic = toc();
    if( obj != NULL ) {
      obj->init();
      obj->run();
      time += toc() - tic;
      logZ = real(obj->logZ());
      maxdiff = obj->MaxDiff();
      q = allBeliefs();
    };
  }

  void calcErrs( const TestAI &x ) {
    err.clear();
    err.reserve( q.size() );
    for( size_t i = 0; i < q.size(); i++ )
      err.push_back( dist( q[i], x.q[i], Prob::DISTLINF ) );
    //err.push_back( dist( q[i], x.q[i], Prob::DISTTV ) );
  }

  void calcErrs( const vector<Factor> &x ) {
    err.clear();
    err.reserve( q.size() );
    for( size_t i = 0; i < q.size(); i++ )
      err.push_back( dist( q[i], x[i], Prob::DISTLINF ) );
    //err.push_back( dist( q[i], x[i], Prob::DISTTV ) );
  }

  double maxErr() { 
    return( *max_element( err.begin(), err.end() ) );
  }
        
  double avgErr() { 
    return( accumulate( err.begin(), err.end(), 0.0 ) / err.size() );
  }
};

pair<string, Properties> parseMethod( const string &_s, const map<string,string> & aliases ) {
  string s = _s;
  if( aliases.find(_s) != aliases.end() )
    s = aliases.find(_s)->second;

  pair<string, Properties> result;
  string & name = result.first;
  Properties & opts = result.second;

  string::size_type pos = s.find_first_of('[');
  name = s.substr( 0, pos );
  if( pos == string::npos )
    throw "Malformed method";
  size_t n = 0;
  for( ; n < sizeof(DAINames) / sizeof(string); n++ )
    if( name == DAINames[n] )
      break;
  if( n == sizeof(DAINames) / sizeof(string) )
    throw "Unknown inference algorithm";

  stringstream ss;
  ss << s.substr(pos,s.length());
  ss >> opts;

  return result;
}

double clipdouble( double x, double minabs ) {
  if( fabs(x) < minabs )
    return minabs;
  else
    return x;
}


